/*
 * Dynamic_Calibration.c
 *
 *  Created on: Jan 26, 2013
 *      Author: Alexander Kozitsky
 */
#include "Dynamic_Calibration.h"
#include <stdio.h>
#include <stdio.h>
#include <string.h>
#include "msp430f23x0.h"

#ifdef UART_PRINTF
int fputc(int _c, register FILE *_fp);
int fputs(const char *_ptr, register FILE *_fp);
#endif

static u16_t Maximum_Time_Value_Sample_Circular_Buffer_Age[NUMBER_OF_MAXIMUM_VALUES_TO_TRACK];
static u16_t Maximum_Time_Value_Sample_Circular_Buffer[NUMBER_OF_MAXIMUM_VALUES_TO_TRACK];
static u16_t Delay_Buffer_Line[DELAY_LINE_NUMBER_OF_SAMPLES];
static s16_t Maximum_Sample_Circular_Buffer_Location = -1;
static u16_t Delay_Buffer_Position = 0;
u08_t Card_Detected = 0;

//Performs "dynamic calibration" to compensate for power supply drifts to prevent false positives
u08_t Automatic_Calibration(u16_t time, u16_t threshold_offset)
{
	u08_t perform_read = 0;
	static u08_t init = 1;
	u16_t max_time_value;
	static u16_t ignore_samples = 0;
#ifdef UART_PRINTF  //useful for EVm debugging, will send UART data over the serial connection to a computer terminal
	static long time_log = 0;
	P3SEL |= BIT4 + BIT5;					// P3.4, P3.5 - UART mode
	P3DIR |= BIT4;							// P3.4 - output direction
	UCA0CTL1 |= UCSWRST;					// disable UART
	UCA0CTL0 = 0x00;
	UCA0CTL1 |= UCSSEL_2;      				// SMCLK
	UCA0BR0 = 118;            				// 115200
	UCA0BR1 = 0;
	UCA0MCTL = 0;               			// Modulation UCBRSx = 5
	UCA0CTL1 &= ~UCSWRST;       			// **Initialize USCI state machine**
	IE2 |= UCA0RXIE;            			// Enable USCI_A0 RX interrupt
#endif

	if (init)								// runs only once on startup to initialize
	{
		init = 0;
		Set_All_Values_In_Delay_Buffer_To (SniffCardTimeThreshold - threshold_offset); //needed to fill the delay line, before new samples arrive
	}
	if(Card_Detected == 1)
	{
		Card_Detected = 0;
		Fill_Delay_Buffer_With_Oldest_Value (); //needed to fill the delay line, with oldest value; newer values are unreliable to calibrat to as a card was presented and was read
		ignore_samples = IGNORE_SAMPLES_AFTER_READ;
#ifdef UART_PRINTF
		printf("\n\r*********CARD DETECTED");
#endif
	}
	if (ignore_samples)						// ignores this many samples after a read has been performed to prevent corrupting the calibration routine with wrong values
	{
		ignore_samples--;
		return 0; 							// ignore this sample and indicate that no detection occured
	}

	max_time_value = Store_Value_In_Min_Buffer (Next_Value_In_Delay_Buffer(time));  //delay line used here, whole function returns the maximum value for MOVING_WINDOW_WHERE_MAXIMUM_SAMPLE_IS_COMPENSATED_FOR

	SniffCardTimeThreshold = max_time_value + threshold_offset;
#ifdef UART_PRINTF
	printf("\n\r%6ld: Time: %d, Use Time: %d + %d -> Sniff Threshold: %d  ", time_log, time, max_time_value, threshold_offset, SniffCardTimeThreshold);
	time_log++;
#endif
	if (time >= SniffCardTimeThreshold)
	{
		perform_read = 1;
#ifdef UART_PRINTF
		printf("\n\rPERFORM READ*****");
#endif
	}


	return perform_read;
}

//Stores the value in buffer if it is smaller than the biggest
//Returns the smallest value
u16_t Store_Value_In_Min_Buffer (u16_t time)
{
	u16_t index;
	u16_t max_value = 0;

	if (Maximum_Sample_Circular_Buffer_Location < 0) // init
	{
		Maximum_Sample_Circular_Buffer_Location = 0;
		Maximum_Time_Value_Sample_Circular_Buffer[Maximum_Sample_Circular_Buffer_Location] = time;
		Maximum_Time_Value_Sample_Circular_Buffer_Age[Maximum_Sample_Circular_Buffer_Location] = 0;
		for(index = 1; index < NUMBER_OF_MAXIMUM_VALUES_TO_TRACK; index++)
		{
			Maximum_Time_Value_Sample_Circular_Buffer[index] = 0;
			Maximum_Time_Value_Sample_Circular_Buffer_Age[index] = 10000; // initialize the buffer
		}
	}
	else if (Maximum_Time_Value_Sample_Circular_Buffer[Maximum_Sample_Circular_Buffer_Location] <= time)
	{	//Replace the the immediate previous value with a new bigger and recent value
		Maximum_Time_Value_Sample_Circular_Buffer[Maximum_Sample_Circular_Buffer_Location] = time;
		Maximum_Time_Value_Sample_Circular_Buffer_Age[Maximum_Sample_Circular_Buffer_Location] = 0;
	}
	else
	{	//add a new point on a circular buffer
		if (Maximum_Sample_Circular_Buffer_Location >= NUMBER_OF_MAXIMUM_VALUES_TO_TRACK)
		{
			Maximum_Sample_Circular_Buffer_Location = 0;
		}
		else
		{
			Maximum_Sample_Circular_Buffer_Location++;
		}
		Maximum_Time_Value_Sample_Circular_Buffer[Maximum_Sample_Circular_Buffer_Location] = time;
		Maximum_Time_Value_Sample_Circular_Buffer_Age[Maximum_Sample_Circular_Buffer_Location] = 0;
	}

	// find the maximum value in the buffer, ignoring the aged out values (values with age larger than MOVING_WINDOW_WHERE_MAXIMUM_SAMPLE_IS_COMPENSATED_FOR)
	for(index = 0; index < NUMBER_OF_MAXIMUM_VALUES_TO_TRACK; index++)
	{
		if (Maximum_Time_Value_Sample_Circular_Buffer_Age[index] > MOVING_WINDOW_WHERE_MAXIMUM_SAMPLE_IS_COMPENSATED_FOR)
		{
			continue; // skip outdated values
		}
		if (max_value <= Maximum_Time_Value_Sample_Circular_Buffer[index])
		{
			max_value = Maximum_Time_Value_Sample_Circular_Buffer[index];
		}
		Maximum_Time_Value_Sample_Circular_Buffer_Age[index]++;				//age the time entry
	}

	return (max_value);
}

//Since the delay buffer is circular, this function is needed to handle the extracting of the correct value
//Takes a value and stores it in the place of the sent value
u16_t Next_Value_In_Delay_Buffer(u16_t last_value)
{
	u16_t ret_val;

	if (Delay_Buffer_Position >= DELAY_LINE_NUMBER_OF_SAMPLES)
	{
		Delay_Buffer_Position = 0;
	}
	ret_val = Delay_Buffer_Line[Delay_Buffer_Position];
	Delay_Buffer_Line[Delay_Buffer_Position] = last_value;
	Delay_Buffer_Position++;

	return ret_val;
}

//Set the buffer to all same value
void Set_All_Values_In_Delay_Buffer_To (u16_t value)
{
	u16_t index = 0;
#ifdef UART_PRINTF
	printf("\n\rFlushing delay buffer with value: %d  ", value);
#endif
	for (index = 0; index < DELAY_LINE_NUMBER_OF_SAMPLES; index++)
	{
		Delay_Buffer_Line[index] = value;
	}
}

//This clears the delay lines of new values, since it is expected that they may unreliable to calibrate with (a tag was detected during the process)
void Fill_Delay_Buffer_With_Oldest_Value ()
{
	u16_t index = 0;
	u16_t set_val;

	if (Delay_Buffer_Position >= DELAY_LINE_NUMBER_OF_SAMPLES)
	{
		Delay_Buffer_Position = 0;
	}
	set_val = Delay_Buffer_Line[Delay_Buffer_Position];
#ifdef UART_PRINTF
	printf("\n\rFlushing delay buffer with oldest value: %d  ", set_val);
#endif

	for (index = 0; index < DELAY_LINE_NUMBER_OF_SAMPLES; index++)
	{
		Delay_Buffer_Line[index] = set_val;
	}
}

#ifdef UART_PRINTF
int fputc(int _c, register FILE *_fp)
{
	UartPutChar((u08_t)_c);

  return((u08_t)_c);
}

int fputs(const char *_ptr, register FILE *_fp)
{
  u16_t i, len;

  len = strlen(_ptr);

  for(i=0 ; i<len ; i++)
  {
	  UartPutChar((u08_t) _ptr[i]);
  }

  return len;
}
#endif
